<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Ultimate_AI_Plugin_Activation_Task_Manager
{
    private $dashboard_page_id_option_key = 'ultimate_ai_dashboard_page_id';
    private $activation_page_id_option_key = 'ultimate_ai_activation_page_id';
    private $success_page_id_option_key = 'ultimate_ai_success_page_id';
    private $custom_endpoints = ['single_tool', 'documents', 'home', 'ai_writing_tools', 'ai_chatbot', 'ai_images', 'ai_code', 'chat_assistant', 'manage_subscription', 'user_settings', 'admin_settings', 'user_billing', 'user_subscription', 'admin_subscription', 'subscriptions', 'user_management', 'ai_templates', 'create_template', 'categories', 'analytics', 'faq', 'pricing', 'orders', 'success', 'cancel', 'create_subscription', 'text_to_speech', 'blog_wizard', 'editor', 'ai_vision', 'themes'];

    public function __construct()
    {
        register_activation_hook(ULTIMATE_AI_PLUGIN_ROOT.'ultimate_ai.php', [$this, 'ultimate_ai_run_regiser_activation_methods']);
        add_filter('template_include', [$this, 'ultimate_ai_load_dashboard_template'], 9999);
        add_filter('template_include', [$this, 'ultimate_ai_load_activation_template'], 999);
        add_action('init', [$this, 'ultimate_ai_register_custom_post_type']);
        add_action('init', [$this, 'ultimate_ai_add_custom_endpoint']);
        add_action('init', [$this, 'ultimate_ai_category_taxonomy']);
    }

    public function add_new_role()
    {
        add_role('uai_dashboard_viewer', 'Admin dashboard viewer', ['creator' => 'ultimate_ai']);
    }

    public function ultimate_ai_add_custom_endpoint()
    {
        foreach ($this->custom_endpoints as $endpoint) {
            add_rewrite_endpoint($endpoint, EP_PAGES);
        }
    }

    public function ultimate_ai_run_regiser_activation_methods()
    {
        $this->ultimate_ai_create_dashboard_page();
        $this->ultimate_ai_create_success_page();
        $this->ultimate_ai_create_activation_page();
        $this->create_subscribers_table();
        $this->ultimate_ai_setup_default_template();
        $this->add_new_role();
    }

    public function ultimate_ai_setup_default_template()
    {
        $Template_Manager = new Ultimate_AI_Template_Manager();
        $Default_Template_Manager = new Ultimate_AI_Default_Template_Manager();

        foreach ($Default_Template_Manager->get_default_template() as $template) {
            if (!$Template_Manager->ultimate_ai_template_name_exist_by('name', $template['name'])) {
                $Template_Manager->ultimate_ai_create_template(
                    $template['name'],
                    $template['input_fields'],
                    $template['prompt'],
                    $template['description'],
                    $template['key'],
                    $template['icon_url'],
                    $template['is_featured']
                );
            }
        }
    }

    // Admin functions
    private function create_subscribers_table()
    {
        global $wpdb;
        $query = 'CREATE TABLE IF NOT EXISTS '.$wpdb->prefix.'ultimate_ai_subscribers (
            id int(11)          NOT NULL auto_increment,
            user_id             int(200) NOT NULL,
            user_name           varchar(50) NOT NULL,
            subscription_id     varchar(200) NOT NULL,
            subscription_price  float(10) NOT NULL,
            subscription_recurrence  float(10) NOT NULL,
            purchased_at        DATETIME NOT NULL,
            expire_at           DATETIME NOT NULL,
            words_remaining     int(200) NOT NULL,
            images_remaining    int(200) NOT NULL,
            is_active           boolean NOT NULL default 0,
            UNIQUE KEY id (id)
        )';
        $wpdb->query($query);
    }

    public function ultimate_ai_register_custom_post_type()
    {
        // open_ai result
        register_post_type(
            'ultimate_ai_history',
            [
              'labels' => ['name' => esc_html__('Ultimate AI History', ULTIMATE_AI_SLUG)],
              'public' => false,
            ]
        );

        // template
        register_post_type(
            'uai_cpt_template',
            [
              'labels' => ['name' => esc_html__('Ultimate AI Template', ULTIMATE_AI_SLUG)],
              'public' => false,
            ]
        );

        // Chat assistant
        register_post_type(
            'uai_cpt_assistant',
            [
              'labels' => ['name' => esc_html__('Ultimate AI Chat Assistant', ULTIMATE_AI_SLUG)],
              'public' => false,
            ]
        );
        // Faq
        register_post_type(
            'uai_cpt_faq',
            [
              'labels' => ['name' => esc_html__('Ultimate AI FAQ', ULTIMATE_AI_SLUG)],
              'public' => false,
            ]
        );
    }

    public function ultimate_ai_create_dashboard_page()
    {
        $page_id = get_option($this->dashboard_page_id_option_key);

        // if no page id available or page not available then create the page
        if (!$page_id || !get_post_status($page_id)) {
            $page_id = wp_insert_post([
              'post_title' => esc_html__('Dashboard', ULTIMATE_AI_SLUG),
              'post_content' => '[Dashboard page]',
              'post_name' => 'Dashboard',
              'post_type' => 'page',
              'post_status' => 'publish',
              '',
            ]);

            if (!is_wp_error($page_id)) {
                update_option($this->dashboard_page_id_option_key, $page_id, true);
            }
        }
    }

    public function ultimate_ai_create_success_page()
    {
        $page_id = get_option($this->success_page_id_option_key);

        // if no page id available or page not available then create the page
        if (!$page_id || !get_post_status($page_id)) {
            $page_id = wp_insert_post([
              'post_title' => esc_html__('Success', ULTIMATE_AI_SLUG),
              'post_content' => '[uaisuccess]',
              'post_name' => 'Success',
              'post_type' => 'page',
              'post_status' => 'publish',
            ]);

            if (!is_wp_error($page_id)) {
                update_option($this->success_page_id_option_key, $page_id, true);
            }
        }
    }

    public function ultimate_ai_create_activation_page()
    {
        $page_id = get_option($this->activation_page_id_option_key);

        // if no page id available or page not available then create the page
        if (!$page_id || !get_post_status($page_id)) {
            $page_id = wp_insert_post([
              'post_title' => esc_html__('Activation', ULTIMATE_AI_SLUG),
              'post_content' => '[uaiactivation]',
              'post_name' => 'Activation',
              'post_type' => 'page',
              'post_status' => 'publish',
            ]);

            if (!is_wp_error($page_id)) {
                update_option($this->activation_page_id_option_key, $page_id, true);
            }
        }
    }

    public function ultimate_ai_load_dashboard_template($template)
    {
        $page_id = get_option($this->dashboard_page_id_option_key);

        if (is_page($page_id) || is_page('demo')) { // demo page need to delete
            $file = ULTIMATE_AI_PLUGIN_ROOT.'template-dashboard.php';
            if (file_exists($file)) {
                return $file;
            }
        }

        return $template;
    }

    public function ultimate_ai_load_activation_template($template)
    {
        $page_id = get_option($this->activation_page_id_option_key);

        if (is_page($page_id)) {
            $file = ULTIMATE_AI_PLUGIN_ROOT.'activation.php';
            if (file_exists($file)) {
                return $file;
            }
        }

        return $template;
    }

    public function ultimate_ai_category_taxonomy()
    {
        $labels = [
          'name' => _x('Ultimate AI ', 'general name'),
          'singular_name' => _x('Ultimate AI', 'singular name name'),
          'search_name' => __('Search Ultimate AI'),
          'all_items' => __('All Ultimate AI'),
          'parent_item' => __('Parent Ultimate AI'),
          'parent_item' => __('Parent Ultimate AI'),
          'parent_item_colon' => __('Parent Ultimate AI:'),
          'edit_item' => __('Edit Ultimate AI'),
          'update_item' => __('Update Ultimate AI'),
          'add_new_item' => __('Add New Ultimate AI'),
          'new_item_name' => __('New Course Ultimate AI'),
          'menu_name' => __('Ultimate AI'),
        ];

        $args = [
          'labels' => $labels,
          'hierarchical' => true,
          'show_ui' => true,
          'show_admin_column' => true,
          'query_var' => true,
          'rewrite' => ['slug' => 'ultimate-ai-taxonomy'],
        ];
        register_taxonomy('ultimate_ai_category_taxonomy', ['post'], $args);
    }
}

new Ultimate_AI_Plugin_Activation_Task_Manager();
